;spectrum analyser for evm
;Peter Martinez G3PLX 20 September 1997
;loadable with Mk2 PLX bootstrap

;range  filsize  filcofs   decmax   Scanwidth Hz
; 0       31     filter0      2        3200        
; 1       63     filter1      4        1600        
; 2      127     filter2      8         800        
; 3      255     filter3     16         400        
; 4      511     filter4     32         200        
; 5     1023     filter5     64         100        

;memory map  P    X        Y              size
;     00     p   var      var
;   $100     p   rom      rom             $100
;   $200     p    -       Hanning window  $200
;   $400         fftI     fftQ            $200
;   $600         Intermediate buffer      $200
;   $800         Ifilter  Qfilter         $400
;   $C00         sin      cos             $100
;   $D00          -        -              $300
;  $1000         InBuf    OutBuf          $200
;  $1200          -       filter coeffs   $800
;  $1A00

;registers used:
;   R3   output buffer tail pointer       ;interrupt driven
;   N3   output buffer head pointer
;   R7   input buffer head pointer        ;interrupt driven
;   N7   input buffer tail pointer
;   M4   set always to $FFFF for linear addressing

;Data bytes from PC:
;   00DDDDDD  : shift DDDDDD left into 24bit stack
;   0100DDDD  : set codec left input gain to DDDD (*1.5dB)
;   10000DDD  ; set decimation ratio, filter, and filter size (0..5 only)
;   11DDDDDD  ; shift DDDDDD into 24bit stack and move stack to CentreFreq
; i.e. it takes 4 bytes to set a frequency. 

        include 'ioequ.asm'
        include 'codequ.asm'

ctlwd0          equ     nopreamp+highpass+samprate8+stereo+data16
ctlwd1          equ     immed3state+xtal2select+bits64+codecmaster

outputset       equ     phoneson+lineouton+(leftattn*63)+(rightattn*63)

inputset        equ     mikeselect+(15*monitorattn)+(leftgain*1)+(rightgain*0)

pi       equ    3.14159265358979323846
bdiv     equ    $00A       ;baud rate to/from PC $00A=56800 baud =57600-1.4%
sinrom   equ    $100       ;address of sine table in ROM
sample   equ    8000.00    ;audio sample rate
insize   equ    512        ;size of audio input buffer
fftsize  equ    512
fftwide  equ    9          ;=ln2(fftsize)
width    equ    410        ;number of points output (excluding marker)
outsize  equ    512        ;enough for one scan + time to do an fft

          org     X:0
rslot0    dc      0       ;left audio from codec
rslot1    dc      0       ;right audio from codec
rslot2    dc      0       ;output level settings: bit 15 is busy flag
rslot3    dc      0       ;input level settings: bit 21 is overrange flag

ctlset    dc      ctlwd0  ;initialisation values for codec control mode
          dc      ctlwd1
          dc      0
          dc      0

timeslot  dc      0                   ;timeslot ptr counts 0,1,2,3
infreq    dc      1750.0/sample*2     ;frequency of input conversion osc
inthead   dc      interbuf            ;filtered, decimated I/Q into interbuf
filsize   dc      31                  ;size of decimation filter

jmptab    dc      push6
          dc      setgain
          dc      setwidth
          dc      setfreq             ;jump table for commands from PC

          org     Y:0
tslot0    dc      0       ;left audio
tslot1    dc      0       ;right audio
tslot2    dc      0       ;output level settings
tslot3    dc      0       ;input level settings: bit 21 is overrange enable

dtaset    dc      0       ;initialisation values for codec data mode
          dc      0
          dc      outputset
          dc      inputset

stack     dc      0                   ;to build Q24's from bytes
phase     dc      0                   ;phase of input conversion osc
filptr    dc      filter              ;ptr into input filter
decmax    dc      2                   ;decimation count limit
deccount  dc      0                   ;decimation count current
filcofs   dc      filter0             ;select filter coefficients

;width selection table
widtab    dc      filter0             ;pointer to filter coeffs
          dc      31                  ;size of filter
          dc      2                   ;decimation ratio

          dc      filter1
          dc      63
          dc      4

          dc      filter2
          dc      127
          dc      8

          dc      filter3
          dc      255
          dc      16

          dc      filter4
          dc      511
          dc      32

          dc      filter5
          dc      1023
          dc      64

          dc      filter5             ;repeats incase PC sends range 6,7
          dc      1023
          dc      64

          dc      filter5
          dc      1023
          dc      64

          org     P:$000C
          jsr    <ssirx               ;SSI rcv data default handler
          nop
          jsr    <ssirx               ;ssi rcv data error handler
          nop

          org     P:$0018
; feed output buffer to PC via SCI    ;fast interrupt
          movep   Y:(R3)+,X:M_STXH
          nop

          org     P:$0020             ;can go here cos not using hostport
ssirx     bchg    #0,X:timeslot       ;toggle bit0
          jcs    <oddslot             ;skip if it was 1
evenslot  btst    #1,X:timeslot       ;test bit1
          jcs    <slot2               ;skip if it was 1
slot0     movep   X:M_RX,X:(R7)+      ;get left input from codec to inbuf
          movep   X:M_RX,X:rslot0     ;and to rslot0 for codec init
          movep   Y:tslot2,X:M_TX     ;put output setting to codec
          rti
oddslot   bchg    #1,X:timeslot       ;toggle bit1
          jcs    <slot3               ;skip if it was 1
slot1     movep   X:M_RX,X:rslot1     ;get right input from codec
          movep   Y:tslot3,X:M_TX     ;put input settings to codec
          rti
slot2     movep   X:M_RX,X:rslot2     ;get output settings from codec
          movep   Y:tslot0,X:M_TX     ;put left audio out to codec
          rti
slot3     movep   X:M_RX,X:rslot3     ;get input settings from codec
          movep   Y:tslot1,X:M_TX     ;put right audio out to codec
          rti

          org    Y:$200
freq      equ     2.0*pi/@cvf(fftsize)
hantab    equ    *                    ;Hanning seems better than Hamming
count     set    0
          dup    fftsize
          dc     0.5-0.5*@cos(@cvf(count)*freq)
count     set    count+1
          endm

          org    L:$400
fftbuf    dsm    fftsize              ;I and Q fft computation

          org    L:$600
interbuf  dsm    fftsize              ;bandpass I/Q after downconvert

          org    L:$800
filter    dsm    1024                 ;I and Q values at 8 kHz

          org    X:$C00
sintab    equ    *
count     set    0
          dup    fftsize/2
          dc     -@cos(@cvf(count)*freq)
count     set    count+1
          endm

          org    Y:$C00
costab    equ    *
count     set    0
          dup    fftsize/2
          dc     -@sin(@cvf(count)*freq)
count     set    count+1
          endm

          org    X:$1000
inbuf     dsm    insize                   ;input raw audio samples

          org    Y:$1000
outbuf    dsm    outsize                  ;output serial data buffer

filter0   equ    *
      include    '31F1600.COF'            ;+/-1600Hz: decimate=2

filter1   equ    *
      include    '63F800.COF'             ;+/-800Hz:  decimate=4

filter2   equ    *
      include    '127F400.cof'            ;+/-400Hz:  decimate=8

filter3   equ    *
      include    '255F200.cof'            ;+/-200Hz: decimate=16

filter4   equ    *
      include    '511F100.cof'            ;+/-100Hz: decimate=32

filter5   equ    *
      include    '1023F50.cof'            ;+/-50Hz: decimate=64

        org     P:$40
start   ori     #3,MR                   ;disable interrupts
        move    #$FFFF,M4
        bset    #2,OMR                  ;turn ROMs on now
        move    #outbuf,R3              ;output tail ptr
        move    R3,N3                   ;output head ptr
        move    #outsize-1,M3           ;size of output buffer
        move    #inbuf,R7               ;input head ptr
        move    R7,N7                   ;input tail ptr
        movec   #insize-1,M7            ;size of input buffer
        move    Y:decmax,A1
        move    A1,Y:deccount
        clr     A
        movep   #$1C0009,X:M_PCTL ;40 Mhz, disable clock output
        movep   A1,X:M_BCR        ;no waits
	movep   #$4303,X:M_CRA    ;40MHz/16 = 2.5MHz SCLK, WL=16 bits, 4W/F
        movep   #$BB30,X:M_CRB    ;rie,re,te, tie not needed
        movep   #$14,X:M_PCDDR    ;setup pc4, pc2 as outputs for SSI
	movep   #$01E3,X:M_PCC    ;Turn on ssi,sci
        movep   #bdiv,X:M_SCCR    ;set baudrate, internal clocks
        movep   #$1302,X:M_SCR    ;set SCI async, 8-bit, tie
	movep   A1,X:M_PCD        ;codec control mode, hit codec reset
	do      #500,_endel       ;50mS delay
	 rep     #2000            ; (100 us delay)
          nop
_endel  bset    #4,X:M_PCD        ;reset off
        movep   #$6000,X:M_IPR    ;IPR=2 for ssi, =1 for sci
        move    #ctlset,R1        ;source=codec control-settings
        move    #tslot0,R4        ;destn=transmit slots
        do      #4,_eset
         move   X:(R1)+,X0        ;move them
         move   X0,Y:(R4)+
_eset   andi    #$FC,MR           ;SSI interrupt will now fire
waitfr  wait
        jclr    #2,X:M_SR,waitfr  ;wait for rx frame sync
        move    A1,X:timeslot     ;synchronise the timeslot pointer
wait0   wait
        jset    #18,X:rslot0,wait0   ;wait for CLB=0
        bset    #18,Y:tslot0         ;set CLB=1
wait1   wait
        jclr    #18,X:rslot0,wait1   ;wait for CLB=1
        move   #dtaset,R1            ;source=codec data-settings
        move   #tslot0,R4            ;destn=transmit slots
        do      #4,_eset
         move   Y:(R1)+,X0           ;move them
         move   X0,Y:(R4)+
_eset   movep   #$BB00,X:M_CRB       ;set SCLK,FS to input
	bset    #2,X:M_PCD           ;switch to data mode
_cal    wait
        jclr    #15,X:rslot2,_cal    ;wait for calibration to finish
        movep   A1,X:M_STXH          ;send something to start the SCI tx int

;here is the main loop
;if R7<>N7 then process an audio sample into the intermediate buffer
inmix   move    R7,A                      ;get audio buffer head pointer
        move    N7,X0                     ;get audio buffer tail ptr
        cmp     X0,A
        jeq    <dofft                     ;skip if empty
        move	X:infreq,A                ;get frequency
        move    Y:phase,X0                ;get current phase
	add	X0,A	    #>$80,X1      ;add frequency to phase
	move	A1,Y:phase                ;save new phase
	mpy     X0,X1,A     #>$FF,X0      ;shift right 16, remainder to A0
        and     X0,A        #sinrom,X0    ;mask A1 to 8 lsb
        movec   #$FF,M0                   ;sine rom size
	or      X0,A        #$40,N0       ;form sine ROM table address
	move    A1,R0                     ;move ROM address to R0
	move    A0,Y0                     ;move remainder to Y0
	jclr    #23,Y0,round
	move    (R0)+                     ;round up if Y0>$7FFFFF
round	move    Y:(R0+N0),X0              ;X0=coarse cosine
	move    Y:(R0),X1                 ;X1=coarse sine
	mpyr    X1,Y0,A    #pi/256.0,Y1   ;A=sin*rem
       	tfr     X0,A       A,X1           ;fetch cos, move sin*rem
	macr   -X1,Y1,A    N7,R1          ;A=cos-sin*rem*pi/256 (=fine cos)
	mpyr    X0,Y0,B    Y:(R0),X1      ;B=cos*rem, read sin again
	tfr     X1,B       B,X1           ;fetch sin, move cos*rem
	macr    X1,Y1,B    A,Y0           ;B=sin+cos*rem*pi/256 (=fine sine)

        movec   M7,M1                      ;set inbuf size
        move    B,X0                       ;move sine
        move    X:(R1)+,X1                 ;get audio sample
        mpyr    X0,X1,A      R1,N7         ;do I mix, save input tail ptr
        mpyr    Y0,X1,B      Y:filptr,R1   ;do Q mix
        movec   X:filsize,M1
        movec   M4,M2
        move    AB,L:(R1)+                 ;move into filter
        move    Y:deccount,R4              ;get decimation counter
        move    R1,Y:filptr                ;save filter state
        move    (R4)-                      ;count down by 1
        move    R4,A
        tst     A      X:filsize,R2        ;see if it's zero
        jne    <enddec                     ;skip if not yet decimation time
        lua     (R2)-,N2                   ;N2=filsize-1
        tfr     A,B   Y:filcofs,R4         ;zero A,B; point to selected filter
        move    L:(R1)+,X                  ;get first I,Q
        move    Y:(R4)+,Y0                 ;get first coeff
        do      N2,_end
         mac     Y0,X0,B    Y:(R1),X0
         mac     Y0,X1,A    X:(R1)+,X1   Y:(R4)+,Y0       ;do the filter
_end    movec   #fftsize-1,M1
        macr    Y0,X0,B     X:inthead,R1    ;finish the filter
        macr    Y0,X1,A     Y:decmax,R4
        move    AB,L:(R1)+                  ;put I,Q into interbuffer
        move    R1,X:inthead                ;save interbuf headptr
enddec  move    R4,Y:deccount               ;reset decimate count

;if there are 'width+1' empty spaces in outbuf then do a spectrum
dofft   move    R3,A
        move    N3,X1
        sub     X1,A    #outsize,X0    ;subtract head from tail
        jcc     dofft1
        add     X0,A                   ;unwrap if wrapped
dofft1  move    #(width+1),X0
        cmp     X0,A                   ;see if enough space to put a spectrum
        jcs     dopcin                 ;skip if not

;fetch the latest 'fftsize' decimated I/Q samples, window, and fft them
        move    X:inthead,R1          ;R1 source interbuffer
        movec   #fftsize-1,M1         ;wrap inside interbuffer
        move    #fftbuf,R2            ;R2 destination fft arrays
        move    #hantab,R4            ;R4 coefficients
        movec   M4,M2                 ;linear destn and coeffs
        move    L:(R1)+,X             ;first data I/Q
        move    Y:(R4)+,Y1            ;first hamming coeff
        do      #fftsize-1,_end
         mpyr    X0,Y1,B    Y:(R1),X0                 ;window Q,next Q
         mpyr    X1,Y1,A    X:(R1)+,X1    Y:(R4)+,Y1  ;window I,next I,next k
         move    AB,L:(R2)+                           ;move to fftbuf
_end    mpyr     X0,Y1,B
        mpyr     X1,Y1,A
        move     AB,L:(R2)+

         move    #fftsize/2,n0     ;initialize butterflies per group
         move    #1,n2             ;initialize groups per pass
         move    #fftsize/4,n6     ;initialize C pointer offset
         movec   M4,M0             ;initialize A and B address modifiers
         movec   M4,M1             ;for linear addressing
         movec   M4,M5
         movec   #0,M6             ;reverse carry (bit-reversed) addressing

         do      #fftwide,_end_pass
         move    #fftbuf,r0        ;initialize A input pointer
         move    r0,r4             ;initialize A output pointer
         lua     (r0)+n0,r1        ;initialize B input pointer
         move    #costab,r6        ;initialize C input pointer
         lua     (r1)-,r5          ;initialize B output pointer
         move    n0,n1             ;initialize pointer offsets
         move    n0,n4
         move    n0,n5
         ori     #4,mr             ;set to scale down by 2 per pass
         do      n2,_end_grp
         move    x:(r1),x1  y:(r6),y0        ;lookup -sine and 
                                             ; -cosine values
         move    x:(r5),a   y:(r0),b         ;preload data
         asl     A
         move    x:(r6)+n6,x0                ;update C pointer

         do      n0,_end_bfy
         mac     x1,y0,b    y:(r1)+,y1       ;Radix 2 DIT
                                             ;butterfly kernel
         macr    -x0,y1,b   a,x:(r5)+    y:(r0),a
         subl    b,a        x:(r0),b     b,y:(r4)
         mac     -x1,x0,b   x:(r0)+,a  a,y:(r5)
         macr    -y1,y0,b   x:(r1),x1
         subl    b,a        b,x:(r4)+  y:(r0),b
_end_bfy
         move    a,x:(r5)+n5    y:(r1)+n1,y1   ;update A and B pointers
         move    x:(r0)+n0,x1   y:(r4)+n4,y1
_end_grp
         move    n0,b1
         andi    #$F3,mr                        ;scaling normal
         lsr     b   n2,a1     ;divide butterflies per group by two
         lsl     a   b1,n0     ;multiply groups per pass by two
         move    a1,n2
_end_pass

; unscramble, sumsquare, convert to dB, feed to output buffer
         move    N3,R2                   ;get outbuf headptr
         move    M3,M2                   ;set outsize
         move    #(fftbuf+$199),R1       ;1/5th into bit-reversed, folded fft
         move    #0,M1                   ;reverse carry in fft
         move    #fftsize/2,N1           ;reverse carry increment
         move    #$FF,X1
         move    X1,Y:(R2)+              ;mark start of spectrum
         do     #width,endout
          move   X:(R1),X0                 ;get I
          mpy    X0,X0,A     Y:(R1)+N1,X0  ;sqr(I), get Q
          mac    X0,X0,A                   ;sqr(I)+sqr(Q)
          jeq    <zero                     ;skip logarithm if zero
          move    #22,R0
          rep     #22                      ;set -66dB endstop
           norm   R0,A                     ;count 3db steps into R0
          asl     A
          asl     A                        ;shift linear interpolation up
          move    R0,A2                    ;copy 3dB count
          rep     #5
           asr    A                        ;A1=Q24 fraction of 96dB
          asr     A    A1,B1
          add     B,A                      ;A1=A1*1.5   A1=fraction of 64dB
zero      move    A1,Y:(R2)+               ;send to output
endout   move    R2,N3                     ;update output headptr

dopcin   jclr    #M_RDRF,X:M_SSR,donepc    ;skip if nothing from PC
         clr     A
         move    X:M_SRXH,A0               ;get byte from PC
         asl     A
         asl     A      #jmptab,N4         ;shift command bits to A1
         move    A1,R4
         nop
         move    X:(R4+N4),R4              ;get address from jump table
         nop
         jsr     (R4)                    ;do command, with data in 6msb of A0
donepc   jmp     inmix

push6    move    Y:stack,A1              ;shift 6 lsb of byte into stack
         rep     #6
          asl    A
         move    A1,Y:stack
         rts

setgain  move   A0,A1                    ;left 24
         lsr    A      Y:tslot3,X0       ;right 2  ,fetch tslot3
         lsr    A      #$0F,X1           ;move gain bits to A1:$0F0000
         eor    X0,A
         and    X1,A
         eor    X0,A                     ;merge gain data into tslot3
         move   A1,Y:tslot3              ;update tslot3
         rts

setwidth rep    #6
          asl   A                       ;shift width range bits to A1:$000007
         move  #>7,X0
         and    X0,A                    ;mask to range bits
         asl    A         A1,X0
         add    X0,A      #>widtab,X0   ;multiply by 3
         add    X0,A                    ;add base of width table
         move   A1,R4
         nop
         move   Y:(R4)+,X0              ;pick up filter address
         move   X0,Y:filcofs
         move   Y:(R4)+,X0              ;pick up decimation factor
         move   X0,X:filsize
         move   Y:(R4),X0               ;pick up filter size
         move   X0,Y:decmax
         rts

setfreq  jsr    push6                   ;push 6 more bits
         move   Y:stack,X0              ;pick up the stack
         move   X0,X:infreq             ;set the frequency
         rts

         end    start